#ifndef GOBLIN_ARMY_H
#define GOBLIN_ARMY_H

#include <stdlib.h>

// Uncomment to enable debugging messages
//#define DEBUG 1
#if DEBUG
#define dprintf(fmt, ...) printf(fmt, ##__VA_ARGS__)
#else
#define dprintf(fmt, ...) (void)0
#endif

/* Several map sizes are pre-defined:
   - MAP_SIZE 0: small size (32*32)
   - MAP_SIZE 1: medium size (128*60)
   - MAP_SIZE 2: large size (1280*640)
 */
#define MAP_SIZE 0

#if MAP_SIZE == 0
#define WORLD_WIDTH 32
#define WORLD_HEIGHT 32

/* number of "mountains" */
#define NB_ROCKS 5
/* approximate size of each mountain */
#define ROCK_SIZE 5

#define NB_FORGES 1
#define NB_SAWFILLS 1
#define NB_CATHEDRALS 1

#define NB_MINES NB_FORGES * 3
#define NB_FORESTS NB_SAWFILLS * 3
#define NB_QUARRIES NB_CATHEDRALS * 3

#define NB_GOBLIN_PER_FORGE  5
#define NB_GOBLIN_PER_SAWFILL 5
#define NB_GOBLIN_PER_CATHEDRAL 5

#elif MAP_SIZE==1

#define WORLD_WIDTH 128
#define WORLD_HEIGHT 60

#define NB_ROCKS 10
#define ROCK_SIZE 10

#define NB_FORGES 3
#define NB_SAWFILLS 3
#define NB_CATHEDRALS 3

#define NB_MINES NB_FORGES * 3
#define NB_FORESTS NB_SAWFILLS * 3
#define NB_QUARRIES NB_CATHEDRALS * 3

#define NB_GOBLIN_PER_FORGE  10
#define NB_GOBLIN_PER_SAWFILL 10
#define NB_GOBLIN_PER_CATHEDRAL 10


#elif MAP_SIZE==2

#define WORLD_WIDTH 1280
#define WORLD_HEIGHT 640

#define NB_ROCKS 50
#define ROCK_SIZE 20

#define NB_FORGES 10
#define NB_SAWFILLS 10
#define NB_CATHEDRALS 10

#define NB_MINES NB_FORGES * 3
#define NB_FORESTS NB_SAWFILLS * 3
#define NB_QUARRIES NB_CATHEDRALS * 3

#define NB_GOBLIN_PER_FORGE  10
#define NB_GOBLIN_PER_SAWFILL 10
#define NB_GOBLIN_PER_CATHEDRAL 10

#endif

/* how many items fit in a bag */
#define BAG_CAPACITY   10
/* how fast a goblin fills its bag */
#define BAG_FILL_SPEED  1
/* how fast a goblin empties its bag */
#define BAG_EMPTY_SPEED 2

#define min(a, b) (a)<(b) ? (a): (b)
#define max(a, b) (a)>(b) ? (a): (b)


/* describe the current activity of a goblin */
enum goblin_status {
  to_src,   // moving to the src
  filling,  // filling their bag
  to_dest,  // moving to the dest
  emptying, // emptying their bag
};


struct goblin {
  struct location* location;
  int id; // unique identifier
  int bag; // content of the goblin bag [0-10]
  enum goblin_status status; // current activity of the goblin
  struct location* src; // location that fills the bag (mine, forest, quarry)
  struct location* dest;  // location that empties the bag (forge, sawfill, cathedral)

  struct location* cur_dest;  // current destination (either src or dest)
};

struct goblin_list {
  struct goblin* goblin;
  struct goblin_list *next;
};

/* each location may contain one of the following */
enum location_content {
  none,
  mine,
  forest,
  quarry,
  forge,
  sawfill,
  cathedral,  
  rock,
};
  
struct location {
  int i; // i coordinate
  int j; // j coordinate
  struct goblin_list* goblins; // list of goblins currently on the tile
  enum location_content location_content;
};

/* defines the map */
struct world {
  int width;
  int height;
  struct location** map; 	/* 2-D matrix of tiles */
  struct goblin_list* goblins; 
  int nb_goblins;
};

/* return 1 if (i,j) is a valid coordinate in w, or 0 otherwise */
static inline int is_ij_valid(struct world* w, int i, int j) {
  if(i < 0 || i >= WORLD_WIDTH) return 0;
  if(j < 0 || j >= WORLD_HEIGHT) return 0;
  return 1;
}

/* clear the screen (disabled in DEBUG mode) */
static inline void clear_screen() {
#if ! DEBUG
  system("clear");
#endif
}

#endif /* GOBLIN_ARMY_H */
